#include <netinet/in.h>
#include <arpa/inet.h>

#include <err.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "pm_reg.h"

int join_hex_bytes(unsigned char *buf, int len, char *glue)
{
	int i;
	for (i = 0; i < len - 1; i++) {
		printf("%02x", buf[i]);
		printf("%s", glue);
	}
	printf("%02x", buf[i]);
	/* no newline */

	return 0;
}

int dec_mac_address(unsigned char *s, int len)
{
	join_hex_bytes(s, len, ":");
	printf("\n");
    return 0;
}

int dec_hex_dump(unsigned char *buf, int len)
{
    int i;
    for (i = 0; i < len - 1; i++) {
        printf("%02x ", buf[i]);
    }
    printf("%2x\n", buf[i]);

    return 0;
}

int dec_fifo_overflow(unsigned char *s, int len)
{
    printf("%u\n", s[0]);
    return 0;
}

int dec_ip_address(unsigned char *s, int len)
{
    union ip_address_u_tag {
        int ip_address;
        unsigned char buf[sizeof(int)];
    } ip_u;

    ip_u.buf[0] = s[0];
    ip_u.buf[1] = s[1];
    ip_u.buf[2] = s[2];
    ip_u.buf[3] = s[3];

    struct in_addr inaddr;
    inaddr.s_addr = ip_u.ip_address;
    printf("IP address: %s\n", inet_ntoa(inaddr));

    return 0;
}

int dec_1_byte(unsigned char *s, int len)
{
	printf("%d (0x%02x)\n", s[0], s[0]);
	return 0;
}

int enc_1_byte(unsigned char *s, int len, char **value_array, int n_val)
{
	unsigned char i;
	if (is_in_range(value_array[0], 1) < 0) {
		err(1, "range");
	}
	i = (unsigned char) strtol(value_array[0], NULL, 0);
	s[0] = i;
	return 0;
}

int dec_2_bytes(unsigned char *s, int len)
{
	/*
    union short_num_tag {
        unsigned short num;
        unsigned char buf[sizeof(unsigned short)];
        //unsigned char buf[2];
    } short_num;
	short_num.buf[0] = s[0];
	short_num.buf[1] = s[1];
	printf("%d (0x%02x)\n", ntohs(short_num.num), ntohs(short_num.num));
	*/

	short i;
	memcpy(&i, &s[0], sizeof(i));

	printf("%d (0x%04x)\n", ntohs(i), ntohs(i));
	return 0;
}

int enc_2_bytes(unsigned char *s, int len, char **value_array, int n_val)
{
	int i;
	unsigned short n;
    union short_num_tag {
        unsigned short num;
        unsigned char buf[sizeof(unsigned short)];
    } short_num;
	short_num.num = 0;

	if (is_in_range(value_array[0], 2) < 0) {
		err(1, "range");
	}

	n = (unsigned short) strtoul(value_array[0], NULL, 0);
	short_num.num = htons(n);
	for (i = 0; i < sizeof(short); i++) {
		s[i] = short_num.buf[i];
	}

	if (pm_reg_debug) {
		for (i = 0; i < sizeof(short); i++) {
			fprintf(stderr, "enc_2_bytes result: s[%d]: 0x%02x\n", i, s[i]);
		}
	}

	return 0;
}

int dec_4_bytes(unsigned char *s, int len)
{
	/*
    union int_num_tag {
        unsigned int num;
        unsigned char buf[sizeof(unsigned int)];
        //unsigned char buf[2];
    } int_num;
	int_num.buf[0] = s[0];
	int_num.buf[1] = s[1];
	int_num.buf[2] = s[2];
	int_num.buf[3] = s[3];
	*/

	unsigned int i;
	memcpy(&i, &s[0], sizeof(i));

	printf("%u\n", ntohl(i));
	if (pm_reg_debug) {
		join_hex_bytes(s, len, " ");
		printf("\n");
	}
	return 0;
}

int enc_4_bytes(unsigned char *s, int len, char **value_array, int n_val)
{
	int i, n;
    union int_num_tag {
        unsigned int num;
        unsigned char buf[sizeof(unsigned int)];
    } int_num;
	int_num.num = 0;

	if (is_in_range(value_array[0], 4) < 0) {
		err(1, "range");
	}

	n = strtoul(value_array[0], NULL, 0);
	int_num.num = htonl(n);
	for (i = 0; i < sizeof(int); i++) {
		s[i] = int_num.buf[i];
	}

	if (pm_reg_debug) {
		for (i = 0; i < sizeof(int); i++) {
			fprintf(stderr, "enc_4_bytes result: s[%d]: 0x%02x\n", i, s[i]);
		}
	}

	return 0;
}

int dec_5_bytes(unsigned char *s, int len)
{
    union int_num_tag {
        unsigned long long num;
        unsigned char buf[sizeof(unsigned long long)];
    } long_long_num;

	/* initialize with 0's */
	long_long_num.num = 0;

	long_long_num.buf[0] = s[4];
	long_long_num.buf[1] = s[3];
	long_long_num.buf[2] = s[2];
	long_long_num.buf[3] = s[1];
	long_long_num.buf[4] = s[0];

	printf("%llu", long_long_num.num);
	if (pm_reg_debug) {
		printf(" ");
		join_hex_bytes(s, len, " ");
	}
	printf("\n");
	return 0;
}

int enc_5_bytes(unsigned char *s, int len, char **value_array, int n_val)
{
	int i;
    union long_long_num_tag {
        unsigned long long num;
        unsigned char buf[sizeof(unsigned long long)];
    } long_long_num;

	if (is_in_range(value_array[0], 5) < 0) {
		err(1, "overflow");
	}

	/* value may be '0x' prefix */
	long_long_num.num = strtoull(value_array[0], NULL, 0);

	/* we don't have htonll so re-order the number manually */
	s[0] = long_long_num.buf[4];
	s[1] = long_long_num.buf[3];
	s[2] = long_long_num.buf[2];
	s[3] = long_long_num.buf[1];
	s[4] = long_long_num.buf[0];

	if (pm_reg_debug) {
		for (i = 0; i < 5; i++) {
			fprintf(stderr, "enc_5_bytes: s[%d]: 0x%02x\n", i, s[i]);
		}
	}
	return 0;
}

int enc_current_time(unsigned char *s, int len, char **value_array, int n_val)
{
	/* this function does not need any input value
	   because the computer running this program is the time data source
	   unsigned buf[8];
	*/

	time_t now, net_now;
	if (time(&now) < 0) {
		err(1, "time()");
	}
	if (pm_reg_debug) {
		fprintf(stderr, "Unix Epoch: %d\n", (int) now);
	}
	now -= PSD_EPOCH;
	if (pm_reg_debug) {
		fprintf(stderr, "PSD Epoch: %d\n", (int) now);
	}
	if (now > 1073741824 - 1) { /* 1073741824 == 2^30 */
		err(1, "Overflow PSD Time format");
	}

	now = (now << 2);
	net_now = htonl(now);
	memcpy(s, &net_now, 4);
	s[4] = s[5] = s[6] = 0;

	/*
	buf[0] = (now >> 22) & 0xff;
	buf[1] = (now >> 14) & 0xff;
	buf[2] = (now >>  6) & 0xff;
	buf[3] = (now <<  2) & 0xff;
	fprintf(stderr, "s:   %02x %02x %02x %02x\n", s[0], s[1], s[2], s[3]);
	fprintf(stderr, "buf: %02x %02x %02x %02x\n", buf[0], buf[1], buf[2], buf[3]);
	*/
	
	return 0;
}

int dec_current_time(unsigned char *s, int len)
{
	time_t net_current_time;
	time_t host_current_time;
	unsigned char ss_buf[2];
	char  *string_time;

	if (pm_reg_debug) {
		join_hex_bytes(s, len, " ");
		printf(" ");
	}

	memcpy(&net_current_time, s, 4);
	/*
	 * net_current_time = (net_current_time & 0xfcffffff);
	 *
	 * does not need this as result.  Needless two bit will be stripped off
	 * at host_current_time >> 2
	 */
	host_current_time = ntohl(net_current_time);
	host_current_time = (host_current_time >> 2);
	host_current_time += PSD_EPOCH;
	string_time = ctime(&host_current_time);
	string_time[strlen(string_time) - 1] = '\0';
	printf("%s", string_time);

	ss_buf[1]  = (s[5] >> 3);
	ss_buf[1] += ((s[4] & 0x07) << 5);
	ss_buf[0]  = ((s[4] & 0xF8) >> 3);
	ss_buf[0] += ((s[3] & 0x3) << 5);
	printf(" (ss: %8.6f)\n", (256*ss_buf[0] + ss_buf[1])/32768.0);

	return 0;
}

int dec_event_num(unsigned char *s, int len)
{
	int event_num;
	if (pm_reg_debug) {
		join_hex_bytes(s, len, " ");
		printf(" ");
	}
	event_num = (s[2] & 0x7F) * 256 * 256 + s[1] * 256 + s[0];

	printf("%d (XXX unit is 16bit)\n", event_num);
	
	return 0;
}

int dec_status_reg(unsigned char *s, int len)
{
	/* XXX */
	if ((s[1] & 0x80) == 0x80) {
		printf("timing clock supplying\n");
	}
	else if ((s[1] & 0x80) == 0x00) {
		printf("timing clock not supplying\n");
	}

	return 0;
}

int enc_status_reg(unsigned char *s, int len, char **value_array, int n_val)
{
	/* XXX */
	printf("sorry, not yet\n");
	return 0;
}

int dec_lld(unsigned char *s, int len)
{
	unsigned int lld, tmh, tml;

	lld = s[0]*256 + s[1];
	tmh = s[2]*256*256 + s[3]*256 + s[4];
	tml = s[5]*256*256 + s[6]*256 + s[7];

	printf("LLD: %d TMH: %d TML: %d\n", lld, tmh, tml);

	return 0;
}

int enc_lld(unsigned char *s, int len, char **value_array, int n_val)
{
    union int_num_tag {
        unsigned int num;
        unsigned char buf[sizeof(unsigned int)];
    } int_num;

	short lld;
	unsigned int tmh, tml;
	int_num.num = 0;

	if (is_in_range(value_array[0], sizeof(short)) < 0) {
		err(1, "lld overflow");
	}
	if (is_in_range(value_array[1], 3) < 0) {
		err(1, "tmh overflow");
	}
	if (is_in_range(value_array[2], 3) < 0) {
		err(1, "tml overflow");
	}

	lld = (short) strtoul(value_array[0], NULL, 0);
	lld = htons(lld);
	memcpy(&s[0], &lld, sizeof(short));

	tmh = strtoul(value_array[1], NULL, 0);
	int_num.num = tmh;
	s[2] = int_num.buf[2];
	s[3] = int_num.buf[1];
	s[4] = int_num.buf[0];

	tml = strtoul(value_array[2], NULL, 0);
	int_num.num = tml;
	s[5] = int_num.buf[2];
	s[6] = int_num.buf[1];
	s[7] = int_num.buf[0];

	if (pm_reg_debug) {
		fprintf(stderr, "%s %s %s\n",
			value_array[0], value_array[1], value_array[2]);
		fprintf(stderr, "ttl result: %02x %02x %02x %02x %02x %02x %02x %02x\n",
			s[0], s[1], s[2], s[3], s[4], s[5], s[6], s[7]);
	}

	return 0;
}

int dec_foo(unsigned char *s, int len)
{
	return 0;
}

int enc_foo(unsigned char *s, int len, char **value_array, int n_val)
{
	return 0;
}

int default_verify(unsigned char *s, unsigned char *r, register_info *ri)
{
    int i;
    int diff_count = 0;
    for (i = 0; i < ri->length; i++) {
        if (pm_reg_debug) {
            printf("send: %02x recv: %02x\n", s[i], r[i]);
        }
        if (s[i] != r[i]) {
			if (pm_reg_debug) {
            	fprintf(stderr, "DIFF: send: 0x%02x recv: 0x%02x\n", s[i], r[i]);
			}
            diff_count ++;
        }
    }
    if (diff_count > 0) {
        return -1;
    }
    else {
        return 0;
    }
}

int dec_tcp_conn(unsigned char *s, int len)
{
	unsigned char tcp_conn_state;
	unsigned char mask = (1 << 5);
	tcp_conn_state = s[0];

	if (pm_reg_debug) {
		fprintf(stderr, "tcp_conn_state byte: %02x\n", tcp_conn_state);
	}
	if ((tcp_conn_state & mask) == mask) {
		printf("tcp connect state: connected\n");
	}
	else {
		printf("tcp connect state: not connected\n");
	}
	return 0;
}

int dec_gate(unsigned char *s, int len)
{
	unsigned char close_open_byte;
	close_open_byte = (s[3] & 0x80);

	if (pm_reg_debug) {
		join_hex_bytes(s, len, " ");
		printf("\n");
	}

	if (close_open_byte == 0x80) {
		printf("gate state: opened\n");
	}
	else if (close_open_byte == 0x00) {
		printf("gate state: closed\n");
	}

    return 0;
}

int verify_gate(unsigned char *s, unsigned char *r, register_info *ri)
{
	int i;
	int rv = -1;
	unsigned char open_close_byte;

	if (pm_reg_debug) {
		fprintf(stderr, "---> verify_gate\n");
		for (i = 0; i < ri->length; i++) {
			fprintf(stderr, "send: %02x recv: %02x\n", s[i], r[i]);
		}
	}
	open_close_byte = (r[3] & 0x80);
	if (s[3] == 0x00) { /* we send gate close command */
		//if (r[3] == 0x00) {
		if (pm_reg_debug) {
			fprintf(stderr, "open_close_byte: %02x\n", open_close_byte);
		}
		if (open_close_byte == 0x00) {
			rv = 0; /* ok */
		}
		else {
			rv = -1;
		}
	}
	
	/* We have to get the open-close bit because
	   we get 0xc0 (additional ON in the 7th bit)
	   if the gate open succeed and the beam ON */

	if (s[3] == 0x80) { /* we send gate open command */
		//open_close_byte = (r[3] & 0x80);
		if (pm_reg_debug) {
			fprintf(stderr, "open_close_byte: %02x\n", open_close_byte);
		}
		if (open_close_byte == 0x80) {
			rv = 0;
		}
		else {
			rv = -1;
		}
	}

	return rv;
}

int verify_current_time(unsigned char *s, unsigned char *r, register_info *ri)
{
	int i, diff;
	time_t net_send_time, host_send_time, net_recv_time, host_recv_time;

	if (pm_reg_debug) {
		fprintf(stderr, "---> verify_current_time\n");
		for (i = 0; i < ri->length; i++) {
			fprintf(stderr, "send: %02x recv: %02x\n", s[i], r[i]);
		}
	}

	memcpy(&net_send_time, s, 4);
	memcpy(&net_recv_time, r, 4);

	host_send_time = ntohl(net_send_time);
	host_send_time = (host_send_time >> 2);
	host_recv_time = ntohl(net_recv_time);
	host_recv_time = (host_recv_time >> 2);

	diff = host_recv_time - host_send_time;

	if (pm_reg_debug) {
		fprintf(stderr, "time difference: %d (sec)\n", diff);
	}

	/* allowable max time difference is 1 sec */
	if (diff > 1) {
		return -1;
	}
	else {
		return 0;
	}
}
