#include <stdio.h>
#include "pm_reg.h"

int pm_reg_debug = 0;

/*************************************************************************
 * Register Data Structure
 * name:           name of the register.  This name will be used as
 *                 search key.
 * address:        offset address of the register.
 * length:         length of the register in byte size.
 * can_write:      Wheather the register is RO (0) or RW (1).
 * n_val: If the mode is "set", do we need the value to be set?
 *                 For example, we do not need set data if the register is
 *                 set current time because the time data is read from the
 *                 computer.
 * enc:            pointer to a function to encode input numeric data
 *                 to buffer.
 * dec:            pointer to a function to decode return buffer to numeric
 *                 or string data.
 *************************************************************************/

/* decoder */
extern int dec_mac_address  (unsigned char *, int reg_len);
extern int dec_fifo_overflow(unsigned char *, int reg_len);
extern int dec_ip_address   (unsigned char *, int reg_len);
extern int dec_hex_dump     (unsigned char *, int reg_len);
extern int dec_1_byte       (unsigned char *, int reg_len);
extern int dec_2_bytes      (unsigned char *, int reg_len);
extern int dec_4_bytes      (unsigned char *, int reg_len);
extern int dec_5_bytes      (unsigned char *, int reg_len);
extern int dec_event_num    (unsigned char *, int reg_len);
extern int dec_status_reg   (unsigned char *, int reg_len);
extern int dec_lld          (unsigned char *, int reg_len);
extern int dec_gate         (unsigned char *, int reg_len);
extern int dec_tcp_conn     (unsigned char *, int reg_len);

extern int dec_current_time (unsigned char *, int reg_len);
// extern int dec_foo          (unsigned char *, int reg_len);

/* encoder */
extern int enc_1_byte(
	unsigned char *, int reg_len, char **value_array, int n_val);
extern int enc_2_bytes(
	unsigned char *, int reg_len, char **value_array, int n_val);
extern int enc_4_bytes(
	unsigned char *, int reg_len, char **value_array, int n_val);
extern int enc_5_bytes(
	unsigned char *, int reg_len, char **value_array, int n_val);
extern int enc_current_time(
	unsigned char *, int reg_len, char **value_array, int n_val);
extern int enc_status_reg(
	unsigned char *, int reg_len, char **value_array, int n_val);
extern int enc_lld(
	unsigned char *, int reg_len, char **value_array, int n_val);
//extern int enc_foo(
//	unsigned char *, int reg_len, char **value_array, int n_val);

/* verifier */
extern int default_verify(unsigned char *, unsigned char *, register_info *);
extern int verify_gate(unsigned char *, unsigned char *, register_info *);
extern int verify_current_time(unsigned char *, unsigned char *, register_info *);

register_info register_info_list[] = {
	/* name            Addr  Len WR n_val dec_func name      enc_func_name    verify_func_name */
    { "mac_address"  , 0x80,  6, NO,  0,  dec_mac_address  , NULL            , NULL },
	{ "tcp_kif"      , 0x86,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
	{ "tcp_kie"      , 0x88,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
	{ "tcp_eto"      , 0x8A,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
	{ "tcp_dto"      , 0x8C,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
	{ "tcp_msl"      , 0x8E,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
	{ "tcp_rto"      , 0x90,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
    { "ip_address"   , 0x92,  4, NO,  0,  dec_ip_address   , NULL            , NULL },
	{ "tcp_tcpp"     , 0x96,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
	{ "tcp_mss"      , 0x98,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
	{ "udp_udpp"     , 0x9A,  2, NO,  0,  dec_2_bytes      , NULL            , NULL },
    { "fe"           , 0x9C,  1, NO,  0,  dec_fifo_overflow, NULL            , NULL },
    { "fifo_overflow", 0x9C,  1, NO,  0,  dec_fifo_overflow, NULL            , NULL },
    { "event_num"    , 0x9D,  3, NO,  0,  dec_event_num    , NULL            , NULL },
	
	{ "gate"         , 0x180, 4, YES, 1,  dec_gate         , enc_4_bytes     , verify_gate         },
	{ "tcp_conn"     , 0x181, 1, NO,  1,  dec_tcp_conn     , NULL            , NULL                },

	{ "status_reg"   , 0x188, 2, YES, 1,  dec_status_reg   , enc_status_reg  , default_verify      },
    { "crate_num"    , 0x18A, 1, YES, 1,  dec_1_byte       , enc_1_byte      , default_verify      },

    { "pulse_id_cnt" , 0x18B, 5, YES, 1,  dec_5_bytes      , enc_5_bytes     , default_verify      },

    { "lld"          , 0x198, 8, YES, 3,  dec_lld          , enc_lld         , default_verify      },

    { "extclock"     , 0x1A0, 5, YES, 1,  dec_5_bytes      , enc_5_bytes     , default_verify      },

    { "used_hour"    , 0x7C,  4, NO,  0,  dec_4_bytes      , NULL            , default_verify      },
    { "uptime"       , 0x7C,  4, NO,  0,  dec_4_bytes      , NULL            , default_verify      },
    { "current_time" , 0x190, 7, YES, 0,  dec_current_time , enc_current_time, verify_current_time },

	/* !!! FOLLOWING REGISTERS ARE FOR SPARTAN EMULATOR !!! */

	/* name            Addr   Len WR n_val dec_func name enc_func_name verify_func_name */
	{ "ts",            0x1AC,  1, YES, 1, dec_1_byte,  enc_1_byte , default_verify },
	{ "speed",         0x1AD,  1, YES, 1, dec_1_byte,  enc_1_byte , default_verify }, 
	{ "lld_low",       0x1A0,  2, YES, 1, dec_2_bytes, enc_2_bytes, default_verify },
	{ "lld_hi",        0x1A2,  2, YES, 1, dec_2_bytes, enc_2_bytes, default_verify },
	{ "time_low",      0x1A4,  2, YES, 1, dec_2_bytes, enc_2_bytes, default_verify },
	{ "time_hi",       0x1A6,  2, YES, 1, dec_2_bytes, enc_2_bytes, default_verify },
	{ "precount",      0x1A8,  4, YES, 1, dec_4_bytes, enc_4_bytes, default_verify },
//  { "foo",            0x99, 99, YES, 4, dec_foo,     enc_foo    , default_verify },
    { NULL,               0,   0,   0, 0, NULL,        NULL       , NULL           },
	/*terminator*/
}; 
